#!/usr/bin/env python3
# D10 (REAL v3) — B-Circulation & Re-expression with direction-aware edges
# and deterministic, rare backsteps to create circulation variance across loops.
#
# Control remains boolean/ordinal (no curve weights, no RNG in control):
#   - Each loop has S sectors (thin ring). At tick t, a single sector is active.
#   - Active sector index k_i advances when the "advance counter" increases.
#   - On most advances, k_i ← k_i + step_base  (forward); on rare scheduled ticks,
#     k_i ← k_i - step_base  (backstep).
# Eligibility(x,y,t) := (cell in loop i and sector k_i) AND (cooldown==0). accept := eligibility.
#
# Diagnostics:
#   - Direction-aware edges between t-1 and t:
#       if the tick advanced forward: CW = prev[k - step_base], CCW = prev[k + step_base]
#       if the tick advanced backward: CW += 0 ; CCW = prev[k + step_base]  (relative to forward-as-CW)
#   - Re-expression = same-cell 1→1 across consecutive ticks.
# PASS:
#   - all_loops_cw_dominant (cw_edges > ccw_edges per loop), and
#   - Pearson r(circ_net_norm, reexp) >= r_min.

import argparse, csv, hashlib, json, math, sys
from pathlib import Path

# ---------- utils ----------
def ensure_dir(p: Path): p.mkdir(parents=True, exist_ok=True)
def sha256_of_file(p: Path):
    import hashlib
    h = hashlib.sha256()
    with p.open('rb') as f:
        for chunk in iter(lambda: f.read(1<<20), b''):
            h.update(chunk)
    return h.hexdigest()
def write_json(p: Path, obj): ensure_dir(p.parent); p.write_text(json.dumps(obj, indent=2), encoding='utf-8')
def write_csv(p: Path, header, rows):
    ensure_dir(p.parent)
    with p.open('w', newline='', encoding='utf-8') as f:
        w = csv.writer(f); w.writerow(header); w.writerows(rows)

# ---------- geometry ----------
def build_loops(nx, ny, outer_margin, loop_fracs, ring_width, sectors_per_loop):
    cx, cy = (nx-1)/2.0, (ny-1)/2.0
    R_eff = max(min(nx, ny)/2.0 - outer_margin, 8.0)
    loops = []
    for f in loop_fracs:
        r_target = max(1.0, f * R_eff)
        r_s = int(max(1, round(r_target)))
        S = sectors_per_loop
        cells_by_sector = [[] for _ in range(S)]
        for y in range(ny):
            for x in range(nx):
                r = math.hypot(x+0.5 - cx, y+0.5 - cy)
                s = int(max(1, round(r)))
                if r_s <= s < r_s + ring_width:
                    theta = math.atan2((y+0.5 - cy), (x+0.5 - cx))  # [-pi,pi]
                    if theta < 0: theta += 2.0*math.pi
                    k = int((theta / (2.0*math.pi)) * S)
                    if k >= S: k = S-1
                    cells_by_sector[k].append((x, y))
        loops.append({"radius_s": r_s, "width": ring_width, "S": S, "cells_by_sector": cells_by_sector})
    return loops

# ---------- engine ----------
def run_engine(nx, ny, H, loops, duty_ticks, step_base, backstep_every, cooldown_ticks):
    L = len(loops)
    # per-loop state
    k_curr = [0]*L                # current sector index
    adv_idx_prev = [-1]*L         # last advance counter
    step_eff_this_tick = [0]*L    # effective step between (t-1)->t, for edges
    cd = [[0]*nx for _ in range(ny)]

    # commit maps (double-buffer sector counts)
    commit_prev = [[0]*nx for _ in range(ny)]
    commit_now  = [[0]*nx for _ in range(ny)]
    sect_prev = [ [0]*loops[i]["S"] for i in range(L) ]
    sect_now  = [ [0]*loops[i]["S"] for i in range(L) ]

    # accumulators
    total_commits_loop = [0]*L
    cw_edges = [0]*L
    ccw_edges= [0]*L
    reexp_hits  = [0]*L
    reexp_trials= [0]*L

    for t in range(H):
        # clear current tick buffers
        for y in range(ny):
            row = commit_now[y]
            for x in range(nx): row[x] = 0
        for i in range(L):
            S = loops[i]["S"]
            sect_now[i] = [0]*S
            # update active sector with deterministic backstep schedule
            duty = max(1, duty_ticks[i])
            adv_idx = t // duty
            eff = 0
            if adv_idx > adv_idx_prev[i]:
                # choose step sign: forward normally; backstep when (adv_idx % backstep_every[i]) == 0 (and >0)
                if backstep_every[i] > 0 and adv_idx > 0 and (adv_idx % backstep_every[i] == 0):
                    eff = -step_base
                else:
                    eff = step_base
                k_curr[i] = (k_curr[i] + eff) % S
                adv_idx_prev[i] = adv_idx
            step_eff_this_tick[i] = eff  # 0 if no advance this tick

            # accept in active sector, cooldown gate
            count_this_sector = 0
            for (x,y) in loops[i]["cells_by_sector"][k_curr[i]]:
                if cd[y][x] == 0:
                    commit_now[y][x] = 1
                    cd[y][x] = cooldown_ticks[i]
                    total_commits_loop[i] += 1
                    count_this_sector += 1
                else:
                    cd[y][x] = max(0, cd[y][x]-1)
            sect_now[i][k_curr[i]] = count_this_sector

        # re-expression & edges
        if t > 0:
            for i in range(L):
                # reexp
                trials = 0; hits = 0
                for k in range(loops[i]["S"]):
                    for (x,y) in loops[i]["cells_by_sector"][k]:
                        trials += 1
                        if commit_prev[y][x] == 1 and commit_now[y][x] == 1:
                            hits += 1
                reexp_hits[i]   += hits
                reexp_trials[i] += trials

                # direction-aware edges (relative to forward-as-CW)
                S = loops[i]["S"]
                eff = step_eff_this_tick[i]
                if eff > 0:
                    k = k_curr[i]
                    k_fwd_prev = (k - step_base) % S
                    k_rev_prev = (k + step_base) % S
                    c_now = sect_now[i][k]
                    cw_edges[i]  += min(c_now, sect_prev[i][k_fwd_prev])
                    ccw_edges[i] += min(c_now, sect_prev[i][k_rev_prev])
                elif eff < 0:
                    k = k_curr[i]
                    k_rev_prev = (k + step_base) % S
                    c_now = sect_now[i][k]
                    # Backstep contributes to CCW only
                    ccw_edges[i] += min(c_now, sect_prev[i][k_rev_prev])
                # eff == 0: no edges this tick

        # advance tick buffers
        commit_prev, commit_now = commit_now, commit_prev
        sect_prev, sect_now     = sect_now, sect_prev

        # cooldown tick for non-updated cells
        for y in range(ny):
            for x in range(nx):
                if commit_prev[y][x] == 0 and cd[y][x] > 0:
                    cd[y][x] -= 1

    # finalize loop metrics
    loop_metrics = []
    for i in range(L):
        cw, ccw = cw_edges[i], ccw_edges[i]
        denom = (cw + ccw) if (cw + ccw) > 0 else 1
        circ_net_norm = (cw - ccw) / float(denom)
        reexp = (reexp_hits[i] / float(max(1, reexp_trials[i])))
        loop_metrics.append({
            "radius_s": loops[i]["radius_s"],
            "width": loops[i]["width"],
            "S": loops[i]["S"],
            "duty": duty_ticks[i],
            "step_base": step_base,
            "backstep_every": backstep_every[i],
            "cooldown": cooldown_ticks[i],
            "total_commits": total_commits_loop[i],
            "cw_edges": cw, "ccw_edges": ccw,
            "circ_net_norm": circ_net_norm,
            "reexp": reexp
        })
    return loop_metrics

def pearson_r(x, y):
    n = len(x)
    if n < 2: return float('nan')
    mx = sum(x)/n; my = sum(y)/n
    vx = sum((xi-mx)**2 for xi in x); vy = sum((yi-my)**2 for yi in y)
    if vx <= 0 or vy <= 0: return float('nan')
    cov = sum((x[i]-mx)*(y[i]-my) for i in range(n))
    import math
    return cov / math.sqrt(vx*vy)

# ---------- one run ----------
def run_one(manifest_path: Path, diag_path: Path, out_dir: Path):
    ensure_dir(out_dir/'metrics'); ensure_dir(out_dir/'audits'); ensure_dir(out_dir/'run_info')

    manifest = json.loads(manifest_path.read_text(encoding='utf-8'))
    diag     = json.loads(diag_path.read_text(encoding='utf-8'))

    nx = int(manifest["domain"]["grid"]["nx"]); ny = int(manifest["domain"]["grid"]["ny"])
    H  = int(manifest["domain"]["ticks"])

    outer_margin = int(diag["ring"]["outer_margin"])
    ring_width   = int(diag["loops"]["ring_width"])
    sectors      = int(diag["loops"]["sectors"])
    loop_fracs   = diag["loops"]["radius_fracs"]

    step_base    = int(diag["controls"]["step_base"])
    duty_ticks   = diag["controls"]["duty_ticks"]           # per loop
    cooldowns    = diag["controls"]["cooldown_ticks"]       # per loop
    backsteps    = diag["controls"]["backstep_every"]       # per loop (0 => never)

    loops = build_loops(nx, ny, outer_margin, loop_fracs, ring_width, sectors)
    assert len(loops) == len(duty_ticks) == len(cooldowns) == len(backsteps)

    metrics = run_engine(nx, ny, H, loops, duty_ticks, step_base, backsteps, cooldowns)

    circ = [m["circ_net_norm"] for m in metrics]
    rexp = [m["reexp"] for m in metrics]
    r = pearson_r(circ, rexp)

    all_cw = all((m["cw_edges"] > m["ccw_edges"]) for m in metrics)
    r_min = float(diag["tolerances"]["min_corr"])
    PASS = bool(all_cw and (not math.isnan(r)) and (r >= r_min))

    # panel
    rows = []
    for i,m in enumerate(metrics):
        rows.append([i, m["radius_s"], m["width"], m["S"],
                     m["duty"], m["step_base"], m["backstep_every"], m["cooldown"],
                     m["total_commits"], m["cw_edges"], m["ccw_edges"],
                     round(m["circ_net_norm"],6), round(m["reexp"],6)])
    write_csv(out_dir/'metrics'/'em_B_panel.csv',
              ['loop_id','radius_s','ring_width','sectors',
               'duty','step_base','backstep_every','cooldown',
               'total_commits','cw_edges','ccw_edges','circ_net_norm','reexp'],
              rows)

    write_json(out_dir/'audits'/'em_bcirculation.json', {
        "all_loops_cw_dominant": all_cw,
        "pearson_r_circ_vs_reexp": r,
        "r_min": r_min,
        "loops": len(metrics),
        "PASS": PASS
    })

    write_json(out_dir/'run_info'/'hashes.json', {
        "manifest_hash": sha256_of_file(manifest_path),
        "diag_hash":     sha256_of_file(diag_path),
        "engine_entrypoint": f"python {Path(sys.argv[0]).name} --manifest <...> --diag <...> --out <...>"
    })

    print("D10 SUMMARY (v3):", json.dumps({
        "loops": len(metrics),
        "all_cw_dominant": all_cw,
        "pearson_r": r,
        "PASS": PASS,
        "audit_path": str((out_dir/'audits'/'em_bcirculation.json').as_posix())
    }))

# ---------- main ----------
def main():
    ap = argparse.ArgumentParser()
    ap.add_argument('--manifest', required=True)
    ap.add_argument('--diag', required=True)
    ap.add_argument('--out', required=True)
    args = ap.parse_args()
    run_one(Path(args.manifest), Path(args.diag), Path(args.out))

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        out_dir=None
        for i,a in enumerate(sys.argv):
            if a=='--out' and i+1<len(sys.argv):
                out_dir=Path(sys.argv[i+1]); break
        if out_dir:
            ensure_dir(out_dir/'audits')
            write_json(out_dir/'audits'/'em_bcirculation.json',
                       {"PASS": False, "failure_reason": f"{type(e).__name__}: {e}"})
        raise
